# 🚀 Guía de Instalación - Sistema de Licencias GD

## 📋 Descripción General

Este es un sistema completo de gestión de licencias independiente que reemplaza WHMCS para la verificación de licencias de software. Incluye:

- **API RESTful** para verificación de licencias
- **Panel de administración web** completo
- **Base de datos MySQL** con estructura optimizada
- **Sistema de claves locales** para funcionamiento offline
- **Logs detallados** y estadísticas
- **Interfaz moderna** y responsive

## 🖥️ Requisitos del Sistema

### Requisitos Mínimos
- **PHP:** 7.4 o superior
- **MySQL:** 5.7 o superior (o MariaDB 10.2+)
- **Servidor Web:** Apache 2.4+ o Nginx 1.18+
- **Memoria:** 512 MB RAM mínimo
- **Espacio:** 100 MB libres

### Extensiones PHP Requeridas
- `pdo`
- `pdo_mysql`
- `json`
- `mbstring`
- `openssl`

### Extensiones PHP Recomendadas
- `curl` (para verificaciones externas)
- `gd` (para futuras funcionalidades)
- `fileinfo` (para validación de archivos)

## 📁 Estructura de Archivos

```
license-system/
├── api/
│   ├── verify.php          # Endpoint principal de verificación
│   ├── config.php          # Configuración de la API
│   └── functions.php       # Funciones auxiliares
├── admin/
│   ├── index.php           # Dashboard
│   ├── login.php           # Sistema de login
│   ├── clients.php         # Gestión de clientes
│   ├── licenses.php        # Gestión de licencias
│   ├── logs.php           # Visualización de logs
│   ├── logout.php         # Cerrar sesión
│   └── includes/
│       ├── header.php      # Header del panel
│       └── sidebar.php     # Sidebar de navegación
├── assets/
│   ├── css/
│   │   └── admin.css       # Estilos del panel
│   └── js/
│       └── admin.js        # JavaScript del panel
├── database/
│   └── schema.sql          # Estructura de la base de datos
├── logs/                   # Logs del sistema (se crea automáticamente)
├── cache/                  # Cache del sistema (se crea automáticamente)
├── install.php             # Instalador web
└── README.md              # Documentación
```

## 🔧 Instalación Paso a Paso

### Paso 1: Subir Archivos al Servidor

1. **Crear subdominio o directorio:**
   ```bash
   # En tu servidor, crear el directorio
   sudo mkdir -p /var/www/repo.gruposdigitales.com
   
   # O usar un subdirectorio
   sudo mkdir -p /var/www/html/license-system
   ```

2. **Subir archivos via FTP/SFTP o Git:**
   ```bash
   # Via Git (recomendado)
   cd /var/www/repo.gruposdigitales.com
   git clone <repositorio> .
   
   # Via FTP/SFTP
   # Subir todos los archivos al directorio del servidor
   ```

3. **Establecer permisos correctos:**
   ```bash
   sudo chown -R www-data:www-data /var/www/repo.gruposdigitales.com
   sudo chmod -R 755 /var/www/repo.gruposdigitales.com
   sudo chmod -R 775 /var/www/repo.gruposdigitales.com/logs
   sudo chmod -R 775 /var/www/repo.gruposdigitales.com/cache
   ```

### Paso 2: Configurar Base de Datos

1. **Crear base de datos y usuario:**
   ```sql
   CREATE DATABASE license_system_gd CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
   CREATE USER 'license_user'@'localhost' IDENTIFIED BY 'contraseña_segura_aquí';
   GRANT ALL PRIVILEGES ON license_system_gd.* TO 'license_user'@'localhost';
   FLUSH PRIVILEGES;
   ```

### Paso 3: Configurar Servidor Web

#### Para Apache (.htaccess incluido)
```apache
<VirtualHost *:80>
    ServerName repo.gruposdigitales.com
    DocumentRoot /var/www/repo.gruposdigitales.com
    
    <Directory /var/www/repo.gruposdigitales.com>
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog ${APACHE_LOG_DIR}/license_error.log
    CustomLog ${APACHE_LOG_DIR}/license_access.log combined
</VirtualHost>

# Para HTTPS (recomendado en producción)
<VirtualHost *:443>
    ServerName repo.gruposdigitales.com
    DocumentRoot /var/www/repo.gruposdigitales.com
    
    SSLEngine on
    SSLCertificateFile /path/to/certificate.crt
    SSLCertificateKeyFile /path/to/private.key
    
    <Directory /var/www/repo.gruposdigitales.com>
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>
```

#### Para Nginx
```nginx
server {
    listen 80;
    listen 443 ssl;
    server_name repo.gruposdigitales.com;
    root /var/www/repo.gruposdigitales.com;
    index index.php index.html;
    
    # SSL config (si tienes certificado)
    ssl_certificate /path/to/certificate.crt;
    ssl_certificate_key /path/to/private.key;
    
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }
    
    # Bloquear acceso a archivos sensibles
    location ~ /\.(ht|git) {
        deny all;
    }
    
    location /logs/ {
        deny all;
    }
    
    location /cache/ {
        deny all;
    }
}
```

### Paso 4: Ejecutar el Instalador Web

1. **Acceder al instalador:**
   ```
   https://repo.gruposdigitales.com/install.php
   ```

2. **Seguir los pasos del instalador:**
   - ✅ Verificación de requisitos
   - 🗄️ Configuración de base de datos
   - 👤 Creación de usuario administrador
   - ✨ Finalización

3. **Eliminar el instalador por seguridad:**
   ```bash
   sudo rm /var/www/repo.gruposdigitales.com/install.php
   ```

## 🔐 Configuración de Seguridad

### 1. SSL/HTTPS (Obligatorio en Producción)
```bash
# Con Let's Encrypt (gratuito)
sudo certbot --apache -d repo.gruposdigitales.com
```

### 2. Firewall
```bash
# Permitir solo HTTP/HTTPS
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp
sudo ufw enable
```

### 3. Archivo .htaccess (ya incluido)
- Bloquea acceso a directorios sensibles
- Reescribe URLs para el API
- Headers de seguridad

### 4. Configuración de PHP (php.ini)
```ini
; Configuración recomendada para producción
expose_php = Off
display_errors = Off
log_errors = On
error_log = /var/log/php_errors.log
post_max_size = 10M
upload_max_filesize = 10M
max_execution_time = 30
memory_limit = 256M
```

## 🎯 Uso del Sistema

### Panel de Administración
```
URL: https://repo.gruposdigitales.com/admin/
Usuario: admin (por defecto)
Contraseña: (la que configuraste en la instalación)
```

### Endpoint de la API
```
URL: https://repo.gruposdigitales.com/api/verify.php
Métodos: GET, POST
Formatos: JSON, XML, pipes
```

### Ejemplo de Verificación de Licencia
```bash
# Método POST
curl -X POST https://repo.gruposdigitales.com/api/verify.php \
  -d "license_key=gd-12345678" \
  -d "domain=example.com" \
  -d "format=json"

# Método GET
curl "https://repo.gruposdigitales.com/api/verify.php?license_key=gd-12345678&domain=example.com&format=json"
```

### Respuesta Exitosa
```json
{
    "status": "Active",
    "message": "Licencia válida",
    "license_key": "gd-12345678",
    "client_name": "Cliente Ejemplo",
    "client_email": "cliente@example.com",
    "company": "Empresa Example",
    "product_name": "Sistema de Reportes",
    "domain": "example.com",
    "created_at": "2024-01-15 10:30:00",
    "expires_at": "2025-01-15 10:30:00",
    "localkey": "base64_encoded_local_key_here",
    "local_key_expires": "2024-02-14 10:30:00",
    "api_version": "1.0",
    "timestamp": "2024-01-20 15:45:30"
}
```

## 🔧 Integración con tu Sistema de Reportes

### Actualizar archivo license_handler.php
El archivo `includes/license_handler.php` ya está configurado para usar la nueva API:

```php
// La función check_license_with_whmcs ahora usa:
$new_license_api_url = 'https://repo.gruposdigitales.com/api/verify.php';
```

### Probar la Integración
1. Crear un cliente de prueba en el panel admin
2. Asignar una licencia de prueba
3. Probar la verificación desde tu sistema

## 📊 Monitoreo y Mantenimiento

### Logs del Sistema
```bash
# Ver logs de la API
tail -f /var/www/repo.gruposdigitales.com/logs/api.log

# Ver logs del servidor web
tail -f /var/log/apache2/license_error.log
tail -f /var/log/nginx/error.log
```

### Backup Automático
```bash
#!/bin/bash
# Script de backup (backup_license_system.sh)
DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/backups/license_system"

# Crear directorio de backup
mkdir -p $BACKUP_DIR

# Backup de la base de datos
mysqldump -u license_user -p license_system_gd > $BACKUP_DIR/db_$DATE.sql

# Backup de archivos
tar -czf $BACKUP_DIR/files_$DATE.tar.gz /var/www/repo.gruposdigitales.com

# Limpiar backups antiguos (mantener últimos 7 días)
find $BACKUP_DIR -name "*.sql" -mtime +7 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete
```

### Crontab para Mantenimiento
```bash
# Editar crontab
sudo crontab -e

# Agregar estas líneas:
# Backup diario a las 2:00 AM
0 2 * * * /path/to/backup_license_system.sh

# Limpiar logs antiguos cada domingo a las 3:00 AM  
0 3 * * 0 find /var/www/repo.gruposdigitales.com/logs -name "*.log" -mtime +30 -delete

# Limpiar cache cada día a las 4:00 AM
0 4 * * * find /var/www/repo.gruposdigitales.com/cache -name "*" -mtime +1 -delete
```

## 🚨 Solución de Problemas

### Error: "Base de datos no encontrada"
1. Verificar que la base de datos existe
2. Verificar credenciales en `api/config.php`
3. Verificar que MySQL está ejecutándose

### Error: "Permisos insuficientes"
```bash
sudo chown -R www-data:www-data /var/www/repo.gruposdigitales.com
sudo chmod -R 755 /var/www/repo.gruposdigitales.com
```

### Error: "No se puede conectar a la API"
1. Verificar que el servidor web está ejecutándose
2. Verificar configuración de DNS
3. Verificar certificado SSL
4. Verificar firewall

### Error: "Licencia no encontrada"
1. Verificar que la licencia existe en la base de datos
2. Verificar formato de la licencia (debe ser `gd-XXXXXXXX`)
3. Verificar logs en `/logs/api.log`

## 📞 Soporte

Para soporte técnico:
- **Email:** soporte@gruposdigitales.com
- **Sistema:** Panel Admin > Logs para debugging
- **Documentación:** Este archivo y README.md

---

**¡Sistema de Licencias GD listo para producción!** 🎉 