<?php
session_start();

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

require_once '../api/config.php';

$action = $_GET['action'] ?? 'list';
$clientId = $_GET['id'] ?? null;
$message = '';
$error = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo = getDbConnection();
        
        if ($action === 'add') {
            // Agregar nuevo cliente
            $name = trim($_POST['name']);
            $email = trim($_POST['email']);
            $company = trim($_POST['company'] ?? '');
            $phone = trim($_POST['phone'] ?? '');
            $address = trim($_POST['address'] ?? '');
            
            if (empty($name) || empty($email)) {
                throw new Exception('Nombre y email son requeridos');
            }
            
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                throw new Exception('Email inválido');
            }
            
            // Verificar si el email ya existe
            $stmt = $pdo->prepare("SELECT id FROM clients WHERE email = ?");
            $stmt->execute([$email]);
            if ($stmt->fetch()) {
                throw new Exception('Ya existe un cliente con este email');
            }
            
            $stmt = $pdo->prepare("
                INSERT INTO clients (name, email, company, phone, address) 
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([$name, $email, $company, $phone, $address]);
            
            $message = 'Cliente agregado exitosamente';
            $action = 'list';
            
        } elseif ($action === 'edit' && $clientId) {
            // Editar cliente existente
            $name = trim($_POST['name']);
            $email = trim($_POST['email']);
            $company = trim($_POST['company'] ?? '');
            $phone = trim($_POST['phone'] ?? '');
            $address = trim($_POST['address'] ?? '');
            $isActive = isset($_POST['is_active']) ? 1 : 0;
            
            if (empty($name) || empty($email)) {
                throw new Exception('Nombre y email son requeridos');
            }
            
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                throw new Exception('Email inválido');
            }
            
            // Verificar si el email ya existe en otro cliente
            $stmt = $pdo->prepare("SELECT id FROM clients WHERE email = ? AND id != ?");
            $stmt->execute([$email, $clientId]);
            if ($stmt->fetch()) {
                throw new Exception('Ya existe otro cliente con este email');
            }
            
            $stmt = $pdo->prepare("
                UPDATE clients 
                SET name = ?, email = ?, company = ?, phone = ?, address = ?, is_active = ?
                WHERE id = ?
            ");
            $stmt->execute([$name, $email, $company, $phone, $address, $isActive, $clientId]);
            
            $message = 'Cliente actualizado exitosamente';
            $action = 'list';
        }
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Eliminar cliente (GET request con confirmación)
if ($action === 'delete' && $clientId && isset($_GET['confirm'])) {
    try {
        $pdo = getDbConnection();
        
        // Verificar si tiene licencias activas
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM licenses WHERE client_id = ? AND status = 'Active'");
        $stmt->execute([$clientId]);
        $activeLicenses = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        if ($activeLicenses > 0) {
            $error = 'No se puede eliminar: el cliente tiene licencias activas';
        } else {
            // Eliminar cliente (esto también eliminará las licencias por CASCADE)
            $stmt = $pdo->prepare("DELETE FROM clients WHERE id = ?");
            $stmt->execute([$clientId]);
            
            $message = 'Cliente eliminado exitosamente';
            $action = 'list';
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Obtener datos para formularios
$client = null;
if (($action === 'edit' || $action === 'view') && $clientId) {
    try {
        $pdo = getDbConnection();
        $stmt = $pdo->prepare("
            SELECT c.*, 
                   (SELECT COUNT(*) FROM licenses WHERE client_id = c.id) as total_licenses,
                   (SELECT COUNT(*) FROM licenses WHERE client_id = c.id AND status = 'Active') as active_licenses
            FROM clients c 
            WHERE c.id = ?
        ");
        $stmt->execute([$clientId]);
        $client = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$client) {
            $error = 'Cliente no encontrado';
            $action = 'list';
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
        $action = 'list';
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Clientes - Sistema de Licencias GD</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/admin.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="fas fa-users"></i> 
                        <?php
                        switch ($action) {
                            case 'add': echo 'Agregar Cliente'; break;
                            case 'edit': echo 'Editar Cliente'; break;
                            case 'view': echo 'Ver Cliente'; break;
                            default: echo 'Gestión de Clientes'; break;
                        }
                        ?>
                    </h1>
                    
                    <?php if ($action === 'list'): ?>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="?action=add" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Nuevo Cliente
                        </a>
                    </div>
                    <?php endif; ?>
                </div>

                <?php if ($message): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="fas fa-check-circle me-2"></i>
                        <?= htmlspecialchars($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?= htmlspecialchars($error) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($action === 'list'): ?>
                    <!-- Lista de clientes -->
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-list"></i> Lista de Clientes
                            </h5>
                        </div>
                        <div class="card-body">
                            <?php
                            try {
                                $pdo = getDbConnection();
                                $stmt = $pdo->query("
                                    SELECT c.*, 
                                           (SELECT COUNT(*) FROM licenses WHERE client_id = c.id) as total_licenses,
                                           (SELECT COUNT(*) FROM licenses WHERE client_id = c.id AND status = 'Active') as active_licenses
                                    FROM clients c 
                                    ORDER BY c.created_at DESC
                                ");
                                $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
                            ?>
                            
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Nombre</th>
                                            <th>Email</th>
                                            <th>Empresa</th>
                                            <th>Licencias</th>
                                            <th>Estado</th>
                                            <th>Creado</th>
                                            <th>Acciones</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($clients as $client): ?>
                                        <tr>
                                            <td><?= $client['id'] ?></td>
                                            <td>
                                                <strong><?= htmlspecialchars($client['name']) ?></strong>
                                                <?php if ($client['phone']): ?>
                                                <br><small class="text-muted">
                                                    <i class="fas fa-phone"></i> <?= htmlspecialchars($client['phone']) ?>
                                                </small>
                                                <?php endif; ?>
                                            </td>
                                            <td><?= htmlspecialchars($client['email']) ?></td>
                                            <td><?= htmlspecialchars($client['company'] ?: '-') ?></td>
                                            <td>
                                                <span class="badge bg-primary"><?= $client['total_licenses'] ?> total</span>
                                                <?php if ($client['active_licenses'] > 0): ?>
                                                <br><span class="badge bg-success"><?= $client['active_licenses'] ?> activas</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($client['is_active']): ?>
                                                    <span class="badge bg-success">Activo</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Inactivo</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <small><?= date('d/m/Y', strtotime($client['created_at'])) ?></small>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="?action=view&id=<?= $client['id'] ?>" class="btn btn-outline-info" title="Ver">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <a href="?action=edit&id=<?= $client['id'] ?>" class="btn btn-outline-primary" title="Editar">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="licenses.php?client_id=<?= $client['id'] ?>" class="btn btn-outline-success" title="Ver Licencias">
                                                        <i class="fas fa-key"></i>
                                                    </a>
                                                    <?php if ($client['active_licenses'] == 0): ?>
                                                    <a href="?action=delete&id=<?= $client['id'] ?>" 
                                                       class="btn btn-outline-danger" 
                                                       title="Eliminar"
                                                       onclick="return confirm('¿Estás seguro de eliminar este cliente?')">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <?php if (empty($clients)): ?>
                                <div class="text-center py-4">
                                    <i class="fas fa-users fa-3x text-muted mb-3"></i>
                                    <h5 class="text-muted">No hay clientes registrados</h5>
                                    <p class="text-muted">Comienza agregando tu primer cliente</p>
                                    <a href="?action=add" class="btn btn-primary">
                                        <i class="fas fa-plus"></i> Agregar Cliente
                                    </a>
                                </div>
                            <?php endif; ?>
                            
                            <?php
                            } catch (Exception $e) {
                                echo '<div class="alert alert-danger">';
                                echo 'Error al cargar los clientes: ' . htmlspecialchars($e->getMessage());
                                echo '</div>';
                            }
                            ?>
                        </div>
                    </div>

                <?php elseif ($action === 'add' || $action === 'edit'): ?>
                    <!-- Formulario de agregar/editar cliente -->
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-<?= $action === 'add' ? 'plus' : 'edit' ?>"></i>
                                <?= $action === 'add' ? 'Agregar Nuevo Cliente' : 'Editar Cliente' ?>
                            </h5>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="name" class="form-label">Nombre Completo *</label>
                                            <input type="text" class="form-control" id="name" name="name" 
                                                   value="<?= htmlspecialchars($client['name'] ?? $_POST['name'] ?? '') ?>" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="email" class="form-label">Email *</label>
                                            <input type="email" class="form-control" id="email" name="email" 
                                                   value="<?= htmlspecialchars($client['email'] ?? $_POST['email'] ?? '') ?>" required>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="company" class="form-label">Empresa</label>
                                            <input type="text" class="form-control" id="company" name="company" 
                                                   value="<?= htmlspecialchars($client['company'] ?? $_POST['company'] ?? '') ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="phone" class="form-label">Teléfono</label>
                                            <input type="text" class="form-control" id="phone" name="phone" 
                                                   value="<?= htmlspecialchars($client['phone'] ?? $_POST['phone'] ?? '') ?>">
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="address" class="form-label">Dirección</label>
                                    <textarea class="form-control" id="address" name="address" rows="3"><?= htmlspecialchars($client['address'] ?? $_POST['address'] ?? '') ?></textarea>
                                </div>
                                
                                <?php if ($action === 'edit'): ?>
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input type="checkbox" class="form-check-input" id="is_active" name="is_active" 
                                               <?= ($client['is_active'] ?? 1) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="is_active">
                                            Cliente activo
                                        </label>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-save"></i> Guardar
                                    </button>
                                    <a href="clients.php" class="btn btn-secondary">
                                        <i class="fas fa-arrow-left"></i> Volver
                                    </a>
                                </div>
                            </form>
                        </div>
                    </div>

                <?php elseif ($action === 'view' && $client): ?>
                    <!-- Ver detalles del cliente -->
                    <div class="row">
                        <div class="col-lg-8">
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="mb-0">
                                        <i class="fas fa-user"></i> Información del Cliente
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <dl class="row">
                                        <dt class="col-sm-3">Nombre:</dt>
                                        <dd class="col-sm-9"><?= htmlspecialchars($client['name']) ?></dd>
                                        
                                        <dt class="col-sm-3">Email:</dt>
                                        <dd class="col-sm-9">
                                            <a href="mailto:<?= htmlspecialchars($client['email']) ?>">
                                                <?= htmlspecialchars($client['email']) ?>
                                            </a>
                                        </dd>
                                        
                                        <?php if ($client['company']): ?>
                                        <dt class="col-sm-3">Empresa:</dt>
                                        <dd class="col-sm-9"><?= htmlspecialchars($client['company']) ?></dd>
                                        <?php endif; ?>
                                        
                                        <?php if ($client['phone']): ?>
                                        <dt class="col-sm-3">Teléfono:</dt>
                                        <dd class="col-sm-9"><?= htmlspecialchars($client['phone']) ?></dd>
                                        <?php endif; ?>
                                        
                                        <?php if ($client['address']): ?>
                                        <dt class="col-sm-3">Dirección:</dt>
                                        <dd class="col-sm-9"><?= nl2br(htmlspecialchars($client['address'])) ?></dd>
                                        <?php endif; ?>
                                        
                                        <dt class="col-sm-3">Estado:</dt>
                                        <dd class="col-sm-9">
                                            <?php if ($client['is_active']): ?>
                                                <span class="badge bg-success">Activo</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Inactivo</span>
                                            <?php endif; ?>
                                        </dd>
                                        
                                        <dt class="col-sm-3">Fecha de registro:</dt>
                                        <dd class="col-sm-9"><?= date('d/m/Y H:i', strtotime($client['created_at'])) ?></dd>
                                        
                                        <dt class="col-sm-3">Última actualización:</dt>
                                        <dd class="col-sm-9"><?= date('d/m/Y H:i', strtotime($client['updated_at'])) ?></dd>
                                    </dl>
                                    
                                    <div class="d-flex gap-2 mt-3">
                                        <a href="?action=edit&id=<?= $client['id'] ?>" class="btn btn-primary">
                                            <i class="fas fa-edit"></i> Editar
                                        </a>
                                        <a href="licenses.php?client_id=<?= $client['id'] ?>" class="btn btn-success">
                                            <i class="fas fa-key"></i> Ver Licencias
                                        </a>
                                        <a href="clients.php" class="btn btn-secondary">
                                            <i class="fas fa-arrow-left"></i> Volver
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-lg-4">
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="mb-0">
                                        <i class="fas fa-chart-bar"></i> Estadísticas
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <div class="d-flex justify-content-between mb-2">
                                        <span>Total de licencias:</span>
                                        <strong><?= $client['total_licenses'] ?></strong>
                                    </div>
                                    <div class="d-flex justify-content-between mb-2">
                                        <span>Licencias activas:</span>
                                        <strong class="text-success"><?= $client['active_licenses'] ?></strong>
                                    </div>
                                    <div class="d-flex justify-content-between mb-2">
                                        <span>Licencias inactivas:</span>
                                        <strong class="text-muted"><?= $client['total_licenses'] - $client['active_licenses'] ?></strong>
                                    </div>
                                </div>
                            </div>
                            
                            <?php if ($client['total_licenses'] > 0): ?>
                            <div class="card mt-3">
                                <div class="card-header">
                                    <h5 class="mb-0">
                                        <i class="fas fa-key"></i> Acciones Rápidas
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <div class="d-grid gap-2">
                                        <a href="licenses.php?action=add&client_id=<?= $client['id'] ?>" class="btn btn-primary">
                                            <i class="fas fa-plus"></i> Nueva Licencia
                                        </a>
                                        <a href="licenses.php?client_id=<?= $client['id'] ?>" class="btn btn-outline-primary">
                                            <i class="fas fa-list"></i> Ver Todas las Licencias
                                        </a>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 