<?php
/**
 * Configuración de la API del Sistema de Licencias
 * Grupos Digitales - License System
 */

// Configuración de errores
error_reporting(E_ALL);
ini_set('display_errors', 0); // En producción debe ser 0

// Configuración de zona horaria
date_default_timezone_set('America/Mexico_City');

// Configuración de la base de datos
// ESTOS VALORES DEBEN SER CONFIGURADOS DURANTE LA INSTALACIÓN
define('DB_HOST', 'localhost');
define('DB_NAME', 'license_system_db');
define('DB_USER', 'license_user');
define('DB_PASS', 'license_password');
define('DB_CHARSET', 'utf8mb4');

// Configuración de la API
define('API_VERSION', '1.0');
define('API_BASE_URL', 'https://repo.gruposdigitales.com'); // URL base del sistema
define('API_SECRET_KEY', 'gd_license_api_secret_2024'); // Cambiar en producción
define('RATE_LIMIT_REQUESTS', 100); // Peticiones por hora por IP
define('LOCAL_KEY_VALIDITY_DAYS', 30); // Días de validez para claves locales

// Configuración de respuestas
define('DEFAULT_RESPONSE_FORMAT', 'json'); // json, xml, pipes
define('ENABLE_CORS', true);
define('ALLOWED_ORIGINS', '*'); // En producción especificar dominios

// Configuración de logs
define('ENABLE_API_LOGGING', true);
define('LOG_LEVEL', 'INFO'); // DEBUG, INFO, WARNING, ERROR
define('LOG_FILE', __DIR__ . '/../logs/api.log');

// Configuración de seguridad
define('ENABLE_IP_WHITELIST', false);
define('IP_WHITELIST', []); // Array de IPs permitidas
define('ENABLE_DOMAIN_VALIDATION', true);
define('BLOCK_SUSPICIOUS_REQUESTS', true);

// Configuración de cache
define('ENABLE_CACHE', true);
define('CACHE_TTL', 300); // 5 minutos
define('CACHE_DIR', __DIR__ . '/../cache/');

// Estados de licencia permitidos
define('LICENSE_STATUSES', [
    'Active' => 'Licencia activa y funcional',
    'Suspended' => 'Licencia suspendida temporalmente',
    'Expired' => 'Licencia expirada',
    'Cancelled' => 'Licencia cancelada'
]);

// Configuración de formato de licencias
define('LICENSE_PREFIX', 'gd-');
define('LICENSE_LENGTH', 8); // Caracteres después del prefijo
define('LICENSE_PATTERN', '/^gd-[a-z0-9]{8}$/i');

// Configuración de respuestas de error
define('ERROR_MESSAGES', [
    'INVALID_LICENSE' => 'Clave de licencia inválida',
    'LICENSE_NOT_FOUND' => 'Licencia no encontrada',
    'LICENSE_EXPIRED' => 'La licencia ha expirado',
    'LICENSE_SUSPENDED' => 'La licencia está suspendida',
    'DOMAIN_NOT_AUTHORIZED' => 'Dominio no autorizado para esta licencia',
    'RATE_LIMIT_EXCEEDED' => 'Límite de peticiones excedido',
    'MAINTENANCE_MODE' => 'Sistema en mantenimiento',
    'INVALID_REQUEST' => 'Petición inválida',
    'SERVER_ERROR' => 'Error interno del servidor'
]);

// Headers de respuesta por defecto
define('DEFAULT_HEADERS', [
    'Content-Type' => 'application/json; charset=UTF-8',
    'Cache-Control' => 'no-cache, no-store, must-revalidate',
    'Pragma' => 'no-cache',
    'Expires' => '0',
    'X-API-Version' => API_VERSION,
    'X-Powered-By' => 'GD License System'
]);

/**
 * Función para obtener la configuración de la base de datos
 */
function getDatabaseConfig() {
    return [
        'host' => DB_HOST,
        'dbname' => DB_NAME,
        'username' => DB_USER,
        'password' => DB_PASS,
        'charset' => DB_CHARSET,
        'options' => [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
        ]
    ];
}

/**
 * Función para conectar a la base de datos
 */
function getDatabaseConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $config = getDatabaseConfig();
            $dsn = sprintf(
                "mysql:host=%s;dbname=%s;charset=%s",
                $config['host'],
                $config['dbname'],
                $config['charset']
            );
            
            $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
        } catch (PDOException $e) {
            error_log('Database connection error: ' . $e->getMessage());
            throw new Exception('Error de conexión a la base de datos');
        }
    }
    
    return $pdo;
}

/**
 * Alias para compatibilidad con el panel admin
 */
function getDbConnection() {
    return getDatabaseConnection();
}

/**
 * Función para verificar si el sistema está en mantenimiento
 */
function isMaintenanceMode() {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'maintenance_mode'");
        $stmt->execute();
        $result = $stmt->fetch();
        
        return $result && $result['setting_value'] == '1';
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Función para obtener una configuración del sistema
 */
function getSystemSetting($key, $default = null) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        
        return $result ? $result['setting_value'] : $default;
    } catch (Exception $e) {
        return $default;
    }
}

/**
 * Función para crear directorios necesarios
 */
function ensureDirectoriesExist() {
    $dirs = [
        dirname(LOG_FILE),
        CACHE_DIR
    ];
    
    foreach ($dirs as $dir) {
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
    }
}

// Crear directorios al cargar el archivo
ensureDirectoriesExist(); 