<?php
/**
 * Endpoint Principal de Verificación de Licencias
 * Grupos Digitales - License System
 * 
 * Este archivo reemplaza la funcionalidad de WHMCS para verificación de licencias
 * URL: https://repo.gruposdigitales.com/api/verify.php
 */

// Configurar headers de seguridad
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');

require_once __DIR__ . '/functions.php';

try {
    // Validar la solicitud
    validateRequest();
    
    // Manejar solicitudes OPTIONS (CORS preflight)
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit;
    }
    
    // Obtener parámetros de la solicitud
    $licenseKey = sanitizeInput($_POST['licensekey'] ?? $_GET['licensekey'] ?? $_POST['license_key'] ?? $_GET['license_key'] ?? '');
    $domain = sanitizeInput($_POST['domain'] ?? $_GET['domain'] ?? $_SERVER['HTTP_HOST'] ?? '');
    $product = sanitizeInput($_POST['product'] ?? $_GET['product'] ?? 'Sistema de Reportes');
    $ip = getClientIp();
    
    // Log de la solicitud
    logApiRequest('INFO', 'License verification request', [
        'license_key' => $licenseKey,
        'domain' => $domain,
        'ip' => $ip,
        'product' => $product
    ]);
    
    // Validar parámetros requeridos
    if (empty($licenseKey)) {
        sendErrorResponse('INVALID_REQUEST', 400, 'Clave de licencia requerida');
    }
    
    if (empty($domain)) {
        sendErrorResponse('INVALID_REQUEST', 400, 'Dominio requerido');
    }
    
    // Validar formato de la clave de licencia
    if (!validateLicenseFormat($licenseKey)) {
        logLicenseVerification($licenseKey, $domain, $ip, 'Invalid', [], 'Formato de licencia inválido');
        sendErrorResponse('INVALID_LICENSE', 400, 'El formato de la clave de licencia es inválido');
    }
    
    // Obtener datos de la licencia
    $license = getLicenseData($licenseKey);
    
    if ($license === false) {
        sendErrorResponse('SERVER_ERROR', 500, 'Error interno del servidor');
    }
    
    if ($license === null) {
        logLicenseVerification($licenseKey, $domain, $ip, 'Not Found', [], 'Licencia no encontrada');
        sendErrorResponse('LICENSE_NOT_FOUND', 404);
    }
    
    // Verificar si el dominio está autorizado
    if (!isDomainAuthorized($license, $domain)) {
        logLicenseVerification($licenseKey, $domain, $ip, 'Unauthorized Domain', $license, 'Dominio no autorizado');
        sendErrorResponse('DOMAIN_NOT_AUTHORIZED', 403);
    }
    
    // Verificar el estado de la licencia
    $statusCheck = checkLicenseStatus($license);
    
    // Preparar respuesta base
    $response = [
        'status' => $statusCheck['status'],
        'message' => $statusCheck['message'],
        'license_key' => $licenseKey,
        'client_name' => $license['client_name'],
        'client_email' => $license['client_email'],
        'company' => $license['company'] ?: '',
        'product_name' => $license['product_name'],
        'domain' => $domain,
        'ip_address' => $ip,
        'created_at' => $license['created_at'],
        'expires_at' => $license['expires_at'],
        'check_count' => $license['check_count'] + 1,
        'api_version' => API_VERSION,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    // Si la licencia está activa, generar clave local
    if ($statusCheck['status'] === 'Active') {
        try {
            $localKey = createLocalKey($license, $domain);
            $response['localkey'] = $localKey;
            $response['local_key_expires'] = date('Y-m-d H:i:s', strtotime('+' . LOCAL_KEY_VALIDITY_DAYS . ' days'));
            
            logApiRequest('INFO', 'Local key generated successfully', [
                'license_key' => $licenseKey,
                'domain' => $domain
            ]);
        } catch (Exception $e) {
            logApiRequest('WARNING', 'Failed to generate local key', [
                'license_key' => $licenseKey,
                'domain' => $domain,
                'error' => $e->getMessage()
            ]);
            // Continuar sin clave local
        }
    }
    
    // Registrar el log de verificación
    logLicenseVerification($licenseKey, $domain, $ip, $statusCheck['status'], $response);
    
    // Determinar código de respuesta HTTP
    $httpCode = 200;
    switch ($statusCheck['status']) {
        case 'Active':
            $httpCode = 200;
            break;
        case 'Expired':
            $httpCode = 410; // Gone
            break;
        case 'Suspended':
            $httpCode = 423; // Locked
            break;
        case 'Cancelled':
            $httpCode = 410; // Gone
            break;
        default:
            $httpCode = 400;
            break;
    }
    
    // Agregar información adicional según el formato solicitado
    $format = strtolower($_GET['format'] ?? $_POST['format'] ?? DEFAULT_RESPONSE_FORMAT);
    
    if ($format === 'xml') {
        // Para compatibilidad con sistemas que esperan XML como el código original que funciona
        $response['validdomain'] = $domain;
        $response['validip'] = $ip;
        $response['validdirectory'] = '/';
        $response['checkdate'] = date('Ymd');
        
        // Agregar hash MD5 para compatibilidad
        $check_token = time() . md5(mt_rand(100000000, mt_getrandmax()) . $licenseKey);
        $response['md5hash'] = md5(API_SECRET_KEY . $check_token);
        $response['check_token'] = $check_token;
    }
    
    // Enviar respuesta
    sendApiResponse($response, $httpCode, $format);
    
} catch (Exception $e) {
    // Log del error
    logApiRequest('ERROR', 'Unhandled exception in verify.php', [
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => $e->getTraceAsString()
    ]);
    
    // Responder con error genérico
    sendErrorResponse('SERVER_ERROR', 500, 'Error interno del servidor');
}
?> 