<?php
/**
 * Instalador del Sistema de Licencias Independiente
 * Grupos Digitales - License System
 * 
 * Este instalador configura el sistema de licencias que reemplazará a WHMCS
 */

session_start();

// Configuración de errores
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar si ya está instalado
if (file_exists('api/config.php') && !isset($_GET['force'])) {
    showAlreadyInstalledPage();
    exit;
}

// Procesar pasos
$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$action = $_POST['action'] ?? '';

if ($action) {
    switch ($action) {
        case 'check_requirements':
            checkRequirements();
            break;
        case 'configure_database':
            configureDatabase();
            break;
        case 'create_admin':
            createAdmin();
            break;
        case 'complete_installation':
            completeInstallation();
            break;
    }
}

// Mostrar página según el paso
switch ($step) {
    case 1:
        showRequirementsPage();
        break;
    case 2:
        showDatabasePage();
        break;
    case 3:
        showAdminPage();
        break;
    case 4:
        showCompletePage();
        break;
    default:
        header('Location: install.php?step=1');
        exit;
}

/**
 * Muestra la página de sistema ya instalado
 */
function showAlreadyInstalledPage() {
    ?>
    <!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Sistema de Licencias - Ya Instalado</title>
        <script src="https://cdn.tailwindcss.com"></script>
    </head>
    <body class="bg-gray-50">
        <div class="min-h-screen flex items-center justify-center">
            <div class="max-w-md w-full bg-white rounded-lg shadow-lg p-8">
                <div class="text-center">
                    <div class="mx-auto w-16 h-16 bg-yellow-100 rounded-full flex items-center justify-center mb-4">
                        <svg class="w-8 h-8 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                        </svg>
                    </div>
                    <h1 class="text-xl font-bold text-gray-900 mb-2">Sistema Ya Instalado</h1>
                    <p class="text-gray-600 mb-6">
                        El sistema de licencias ya está instalado. Si necesita reinstalar, 
                        elimine el archivo <code class="bg-gray-100 px-2 py-1 rounded">api/config.php</code>.
                    </p>
                    <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                        <p class="text-red-800 text-sm">
                            <strong>¡Importante por seguridad!</strong><br>
                            Elimine este archivo <code>install.php</code> inmediatamente.
                        </p>
                    </div>
                    <a href="admin/" class="inline-block bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors">
                        Ir al Panel Admin
                    </a>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
}

/**
 * Verificar requisitos del sistema
 */
function checkRequirements() {
    $errors = [];
    $warnings = [];

    // Verificar versión de PHP
    if (version_compare(PHP_VERSION, '7.4.0', '<')) {
        $errors[] = 'Se requiere PHP 7.4 o superior. Versión actual: ' . PHP_VERSION;
    }

    // Verificar extensiones requeridas
    $required_extensions = ['pdo', 'pdo_mysql', 'json', 'mbstring', 'openssl'];
    foreach ($required_extensions as $ext) {
        if (!extension_loaded($ext)) {
            $errors[] = "La extensión PHP '$ext' es requerida pero no está disponible.";
        }
    }

    // Verificar extensiones recomendadas
    $recommended_extensions = ['curl', 'gd', 'fileinfo'];
    foreach ($recommended_extensions as $ext) {
        if (!extension_loaded($ext)) {
            $warnings[] = "Se recomienda tener la extensión '$ext' instalada.";
        }
    }

    // Verificar permisos de escritura
    $write_dirs = ['api/', 'logs/', 'cache/', 'admin/'];
    foreach ($write_dirs as $dir) {
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0755, true)) {
                $errors[] = "No se pudo crear el directorio '$dir'.";
                continue;
            }
        }
        if (!is_writable($dir)) {
            $errors[] = "El directorio '$dir' debe tener permisos de escritura.";
        }
    }

    $_SESSION['requirements_check'] = [
        'errors' => $errors,
        'warnings' => $warnings,
        'passed' => empty($errors)
    ];

    if (empty($errors)) {
        header('Location: install.php?step=2');
    } else {
        header('Location: install.php?step=1&error=requirements');
    }
    exit;
}

/**
 * Configurar base de datos
 */
function configureDatabase() {
    $db_host = trim($_POST['db_host'] ?? '');
    $db_name = trim($_POST['db_name'] ?? '');
    $db_user = trim($_POST['db_user'] ?? '');
    $db_pass = trim($_POST['db_pass'] ?? '');

    $errors = [];

    if (empty($db_host)) $errors[] = 'El host de la base de datos es requerido.';
    if (empty($db_name)) $errors[] = 'El nombre de la base de datos es requerido.';
    if (empty($db_user)) $errors[] = 'El usuario de la base de datos es requerido.';

    if (empty($errors)) {
        try {
            $dsn = "mysql:host={$db_host};dbname={$db_name};charset=utf8mb4";
            $pdo = new PDO($dsn, $db_user, $db_pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);

            // Crear tablas usando el archivo schema.sql
            if (!file_exists('database/schema.sql')) {
                throw new Exception('El archivo database/schema.sql no fue encontrado.');
            }

            $sql = file_get_contents('database/schema.sql');
            if (!$sql) {
                throw new Exception('No se pudo leer el archivo database/schema.sql.');
            }

            // Ejecutar SQL
            $pdo->exec($sql);

            $_SESSION['database_config'] = [
                'host' => $db_host,
                'name' => $db_name,
                'user' => $db_user,
                'pass' => $db_pass
            ];

            header('Location: install.php?step=3');
        } catch (Exception $e) {
            $_SESSION['database_errors'] = ['Error de base de datos: ' . $e->getMessage()];
            header('Location: install.php?step=2&error=1');
        }
    } else {
        $_SESSION['database_errors'] = $errors;
        header('Location: install.php?step=2&error=1');
    }
    exit;
}

/**
 * Crear usuario administrador
 */
function createAdmin() {
    $admin_username = trim($_POST['admin_username'] ?? '');
    $admin_email = trim($_POST['admin_email'] ?? '');
    $admin_pass = $_POST['admin_pass'] ?? '';
    $admin_pass_confirm = $_POST['admin_pass_confirm'] ?? '';

    $errors = [];

    if (empty($admin_username)) $errors[] = 'El nombre de usuario es requerido.';
    if (empty($admin_email)) $errors[] = 'El email del administrador es requerido.';
    if (!filter_var($admin_email, FILTER_VALIDATE_EMAIL)) $errors[] = 'El email no es válido.';
    if (empty($admin_pass)) $errors[] = 'La contraseña es requerida.';
    if (strlen($admin_pass) < 8) $errors[] = 'La contraseña debe tener al menos 8 caracteres.';
    if ($admin_pass !== $admin_pass_confirm) $errors[] = 'Las contraseñas no coinciden.';

    if (empty($errors)) {
        $_SESSION['admin_config'] = [
            'username' => $admin_username,
            'email' => $admin_email,
            'password' => password_hash($admin_pass, PASSWORD_DEFAULT)
        ];
        header('Location: install.php?step=4');
    } else {
        $_SESSION['admin_errors'] = $errors;
        header('Location: install.php?step=3&error=1');
    }
    exit;
}

/**
 * Completar instalación
 */
function completeInstallation() {
    try {
        $database = $_SESSION['database_config'] ?? [];
        $admin = $_SESSION['admin_config'] ?? [];

        if (empty($database) || empty($admin)) {
            throw new Exception('Configuración incompleta. Reinicie la instalación.');
        }

        // Generar archivo de configuración
        $api_secret = bin2hex(random_bytes(32));
        $config_content = generateConfigFile($database, $api_secret);
        
        if (!file_put_contents('api/config.php', $config_content)) {
            throw new Exception('No se pudo escribir el archivo api/config.php');
        }

        // Actualizar usuario administrador en la base de datos
        $dsn = "mysql:host={$database['host']};dbname={$database['name']};charset=utf8mb4";
        $pdo = new PDO($dsn, $database['user'], $database['pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);

        $stmt = $pdo->prepare("UPDATE admin_users SET username = ?, email = ?, password = ? WHERE id = 1");
        $stmt->execute([$admin['username'], $admin['email'], $admin['password']]);

        // Limpiar datos de sesión
        session_destroy();

        $_SESSION['installation_complete'] = true;
        header('Location: install.php?step=4&success=1');
    } catch (Exception $e) {
        $_SESSION['installation_error'] = $e->getMessage();
        header('Location: install.php?step=4&error=1');
    }
    exit;
}

/**
 * Generar archivo de configuración
 */
function generateConfigFile($database, $api_secret) {
    return '<?php
/**
 * Configuración del Sistema de Licencias
 * Generado automáticamente el ' . date('Y-m-d H:i:s') . '
 */

// Configuración de errores
error_reporting(E_ALL);
ini_set(\'display_errors\', 0);

// Configuración de zona horaria
date_default_timezone_set(\'America/Mexico_City\');

// Configuración de la base de datos
define(\'DB_HOST\', \'' . addslashes($database['host']) . '\');
define(\'DB_NAME\', \'' . addslashes($database['name']) . '\');
define(\'DB_USER\', \'' . addslashes($database['user']) . '\');
define(\'DB_PASS\', \'' . addslashes($database['pass']) . '\');
define(\'DB_CHARSET\', \'utf8mb4\');

// Configuración de la API
define(\'API_VERSION\', \'1.0\');
define(\'API_SECRET_KEY\', \'' . $api_secret . '\');
define(\'RATE_LIMIT_REQUESTS\', 100);
define(\'LOCAL_KEY_VALIDITY_DAYS\', 30);

// Configuración de respuestas
define(\'DEFAULT_RESPONSE_FORMAT\', \'json\');
define(\'ENABLE_CORS\', true);
define(\'ALLOWED_ORIGINS\', \'*\');

// Configuración de logs
define(\'ENABLE_API_LOGGING\', true);
define(\'LOG_LEVEL\', \'INFO\');
define(\'LOG_FILE\', __DIR__ . \'/../logs/api.log\');

// Configuración de seguridad
define(\'ENABLE_IP_WHITELIST\', false);
define(\'IP_WHITELIST\', []);
define(\'ENABLE_DOMAIN_VALIDATION\', true);
define(\'BLOCK_SUSPICIOUS_REQUESTS\', true);

// Configuración de cache
define(\'ENABLE_CACHE\', true);
define(\'CACHE_TTL\', 300);
define(\'CACHE_DIR\', __DIR__ . \'/../cache/\');

// Estados de licencia permitidos
define(\'LICENSE_STATUSES\', [
    \'Active\' => \'Licencia activa y funcional\',
    \'Suspended\' => \'Licencia suspendida temporalmente\',
    \'Expired\' => \'Licencia expirada\',
    \'Cancelled\' => \'Licencia cancelada\'
]);

// Configuración de formato de licencias
define(\'LICENSE_PREFIX\', \'gd-\');
define(\'LICENSE_LENGTH\', 8);
define(\'LICENSE_PATTERN\', \'/^gd-[a-z0-9]{8}$/i\');

// Configuración de respuestas de error
define(\'ERROR_MESSAGES\', [
    \'INVALID_LICENSE\' => \'Clave de licencia inválida\',
    \'LICENSE_NOT_FOUND\' => \'Licencia no encontrada\',
    \'LICENSE_EXPIRED\' => \'La licencia ha expirado\',
    \'LICENSE_SUSPENDED\' => \'La licencia está suspendida\',
    \'DOMAIN_NOT_AUTHORIZED\' => \'Dominio no autorizado para esta licencia\',
    \'RATE_LIMIT_EXCEEDED\' => \'Límite de peticiones excedido\',
    \'MAINTENANCE_MODE\' => \'Sistema en mantenimiento\',
    \'INVALID_REQUEST\' => \'Petición inválida\',
    \'SERVER_ERROR\' => \'Error interno del servidor\'
]);

// Headers de respuesta por defecto
define(\'DEFAULT_HEADERS\', [
    \'Content-Type\' => \'application/json; charset=UTF-8\',
    \'Cache-Control\' => \'no-cache, no-store, must-revalidate\',
    \'Pragma\' => \'no-cache\',
    \'Expires\' => \'0\',
    \'X-API-Version\' => API_VERSION,
    \'X-Powered-By\' => \'GD License System\'
]);

// Resto de funciones del archivo config.php original
' . file_get_contents('api/config.php') . '
?>';
}

/**
 * Mostrar página de requisitos (Paso 1)
 */
function showRequirementsPage() {
    $errors = $_SESSION['requirements_check']['errors'] ?? [];
    $warnings = $_SESSION['requirements_check']['warnings'] ?? [];
    $show_error = isset($_GET['error']);
    unset($_SESSION['requirements_check']);
    ?>
    <!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Instalador - Sistema de Licencias</title>
        <script src="https://cdn.tailwindcss.com"></script>
    </head>
    <body class="bg-gray-50">
        <div class="min-h-screen py-12">
            <div class="max-w-2xl mx-auto px-4">
                <div class="bg-white rounded-lg shadow-lg p-8">
                    <div class="text-center mb-8">
                        <h1 class="text-3xl font-bold text-gray-900 mb-2">Sistema de Licencias</h1>
                        <p class="text-gray-600">Instalador para el Sistema de Licencias Independiente</p>
                    </div>

                    <div class="mb-6">
                        <div class="flex items-center justify-between mb-2">
                            <span class="text-sm font-medium text-gray-700">Paso 1 de 4: Verificación de Requisitos</span>
                        </div>
                        <div class="w-full bg-gray-200 rounded-full h-2">
                            <div class="bg-blue-600 h-2 rounded-full" style="width: 25%"></div>
                        </div>
                    </div>

                    <?php if ($show_error && !empty($errors)): ?>
                        <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                            <h4 class="text-red-800 font-semibold mb-2">Requisitos no cumplidos:</h4>
                            <ul class="text-red-700 text-sm space-y-1">
                                <?php foreach ($errors as $error): ?>
                                    <li>• <?= htmlspecialchars($error) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($warnings)): ?>
                        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                            <h4 class="text-yellow-800 font-semibold mb-2">Advertencias:</h4>
                            <ul class="text-yellow-700 text-sm space-y-1">
                                <?php foreach ($warnings as $warning): ?>
                                    <li>• <?= htmlspecialchars($warning) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                        <h4 class="text-blue-800 font-semibold mb-2">Requisitos del sistema:</h4>
                        <ul class="text-blue-700 text-sm space-y-1">
                            <li>• PHP 7.4 o superior</li>
                            <li>• Extensiones: PDO, PDO_MySQL, JSON, mbstring, OpenSSL</li>
                            <li>• Extensiones recomendadas: cURL, GD, FileInfo</li>
                            <li>• Permisos de escritura en directorios del sistema</li>
                            <li>• Base de datos MySQL 5.7+ o MariaDB 10.3+</li>
                        </ul>
                    </div>

                    <form method="post">
                        <input type="hidden" name="action" value="check_requirements">
                        <button type="submit" class="w-full bg-blue-600 text-white py-3 px-4 rounded-lg hover:bg-blue-700 transition-colors font-medium">
                            Verificar Requisitos
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
}

/**
 * Mostrar página de configuración de base de datos (Paso 2)
 */
function showDatabasePage() {
    $errors = $_SESSION['database_errors'] ?? [];
    unset($_SESSION['database_errors']);
    $show_error = isset($_GET['error']);
    ?>
    <!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Instalador - Configuración de Base de Datos</title>
        <script src="https://cdn.tailwindcss.com"></script>
    </head>
    <body class="bg-gray-50">
        <div class="min-h-screen py-12">
            <div class="max-w-2xl mx-auto px-4">
                <div class="bg-white rounded-lg shadow-lg p-8">
                    <div class="text-center mb-8">
                        <h1 class="text-3xl font-bold text-gray-900 mb-2">Configuración de Base de Datos</h1>
                        <p class="text-gray-600">Configure la conexión a la base de datos MySQL</p>
                    </div>

                    <div class="mb-6">
                        <div class="flex items-center justify-between mb-2">
                            <span class="text-sm font-medium text-gray-700">Paso 2 de 4: Base de Datos</span>
                        </div>
                        <div class="w-full bg-gray-200 rounded-full h-2">
                            <div class="bg-blue-600 h-2 rounded-full" style="width: 50%"></div>
                        </div>
                    </div>

                    <?php if ($show_error && !empty($errors)): ?>
                        <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                            <h4 class="text-red-800 font-semibold mb-2">Errores de configuración:</h4>
                            <ul class="text-red-700 text-sm space-y-1">
                                <?php foreach ($errors as $error): ?>
                                    <li>• <?= htmlspecialchars($error) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                        <p class="text-blue-800 text-sm">
                            <strong>Importante:</strong> Asegúrese de que la base de datos esté creada. 
                            El instalador creará las tablas automáticamente.
                        </p>
                    </div>

                    <form method="post" class="space-y-6">
                        <input type="hidden" name="action" value="configure_database">
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Host de la Base de Datos *</label>
                            <input type="text" name="db_host" value="localhost" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Nombre de la Base de Datos *</label>
                            <input type="text" name="db_name" placeholder="license_system_db" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Usuario de la Base de Datos *</label>
                            <input type="text" name="db_user" placeholder="license_user" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Contraseña de la Base de Datos</label>
                            <input type="password" name="db_pass" placeholder="contraseña_db"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div class="flex justify-between">
                            <a href="install.php?step=1" class="bg-gray-300 text-gray-700 py-3 px-6 rounded-lg hover:bg-gray-400 transition-colors">
                                Anterior
                            </a>
                            <button type="submit" class="bg-blue-600 text-white py-3 px-6 rounded-lg hover:bg-blue-700 transition-colors">
                                Probar Conexión
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
}

/**
 * Mostrar página de creación de administrador (Paso 3)
 */
function showAdminPage() {
    $errors = $_SESSION['admin_errors'] ?? [];
    unset($_SESSION['admin_errors']);
    $show_error = isset($_GET['error']);
    ?>
    <!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Instalador - Usuario Administrador</title>
        <script src="https://cdn.tailwindcss.com"></script>
    </head>
    <body class="bg-gray-50">
        <div class="min-h-screen py-12">
            <div class="max-w-2xl mx-auto px-4">
                <div class="bg-white rounded-lg shadow-lg p-8">
                    <div class="text-center mb-8">
                        <h1 class="text-3xl font-bold text-gray-900 mb-2">Usuario Administrador</h1>
                        <p class="text-gray-600">Configure la cuenta del administrador principal</p>
                    </div>

                    <div class="mb-6">
                        <div class="flex items-center justify-between mb-2">
                            <span class="text-sm font-medium text-gray-700">Paso 3 de 4: Administrador</span>
                        </div>
                        <div class="w-full bg-gray-200 rounded-full h-2">
                            <div class="bg-blue-600 h-2 rounded-full" style="width: 75%"></div>
                        </div>
                    </div>

                    <?php if ($show_error && !empty($errors)): ?>
                        <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                            <h4 class="text-red-800 font-semibold mb-2">Corrija los siguientes errores:</h4>
                            <ul class="text-red-700 text-sm space-y-1">
                                <?php foreach ($errors as $error): ?>
                                    <li>• <?= htmlspecialchars($error) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <form method="post" class="space-y-6">
                        <input type="hidden" name="action" value="create_admin">
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Nombre de Usuario *</label>
                            <input type="text" name="admin_username" placeholder="admin" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Email *</label>
                            <input type="email" name="admin_email" placeholder="admin@gruposdigitales.com" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Contraseña *</label>
                            <input type="password" name="admin_pass" placeholder="Mínimo 8 caracteres" required minlength="8"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Confirmar Contraseña *</label>
                            <input type="password" name="admin_pass_confirm" placeholder="Repetir contraseña" required minlength="8"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div class="flex justify-between">
                            <a href="install.php?step=2" class="bg-gray-300 text-gray-700 py-3 px-6 rounded-lg hover:bg-gray-400 transition-colors">
                                Anterior
                            </a>
                            <button type="submit" class="bg-blue-600 text-white py-3 px-6 rounded-lg hover:bg-blue-700 transition-colors">
                                Siguiente
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
}

/**
 * Mostrar página de finalización (Paso 4)
 */
function showCompletePage() {
    $installation_complete = $_SESSION['installation_complete'] ?? false;
    $error = $_SESSION['installation_error'] ?? '';
    $show_success = isset($_GET['success']);
    $show_error = isset($_GET['error']);
    
    if (!$installation_complete && !$show_success && !$show_error) {
        ?>
        <!DOCTYPE html>
        <html lang="es">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Instalador - Finalizar</title>
            <script src="https://cdn.tailwindcss.com"></script>
        </head>
        <body class="bg-gray-50">
            <div class="min-h-screen py-12">
                <div class="max-w-2xl mx-auto px-4">
                    <div class="bg-white rounded-lg shadow-lg p-8">
                        <div class="text-center mb-8">
                            <h1 class="text-3xl font-bold text-gray-900 mb-2">Finalizar Instalación</h1>
                            <p class="text-gray-600">¡Ya casi terminamos!</p>
                        </div>

                        <div class="mb-6">
                            <div class="flex items-center justify-between mb-2">
                                <span class="text-sm font-medium text-gray-700">Paso 4 de 4: Finalización</span>
                            </div>
                            <div class="w-full bg-gray-200 rounded-full h-2">
                                <div class="bg-blue-600 h-2 rounded-full" style="width: 100%"></div>
                            </div>
                        </div>

                        <div class="bg-green-50 border border-green-200 rounded-lg p-4 mb-6">
                            <h4 class="text-green-800 font-semibold mb-2">Resumen de configuración:</h4>
                            <ul class="text-green-700 text-sm space-y-1">
                                <li>• ✅ Requisitos del sistema verificados</li>
                                <li>• ✅ Base de datos configurada y tablas creadas</li>
                                <li>• ✅ Usuario administrador configurado</li>
                                <li>• ✅ Sistema listo para generar archivos de configuración</li>
                            </ul>
                        </div>

                        <form method="post">
                            <input type="hidden" name="action" value="complete_installation">
                            <div class="flex justify-between">
                                <a href="install.php?step=3" class="bg-gray-300 text-gray-700 py-3 px-6 rounded-lg hover:bg-gray-400 transition-colors">
                                    Anterior
                                </a>
                                <button type="submit" class="bg-green-600 text-white py-3 px-6 rounded-lg hover:bg-green-700 transition-colors" id="complete-btn">
                                    Completar Instalación
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            <script>
            document.querySelector('form').addEventListener('submit', function() {
                const btn = document.getElementById('complete-btn');
                btn.innerHTML = 'Instalando...';
                btn.disabled = true;
            });
            </script>
        </body>
        </html>
        <?php
        return;
    }

    // Página de instalación completada o error
    ?>
    <!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Instalación - <?= $show_success ? 'Completada' : 'Error' ?></title>
        <script src="https://cdn.tailwindcss.com"></script>
    </head>
    <body class="bg-gray-50">
        <div class="min-h-screen py-12">
            <div class="max-w-2xl mx-auto px-4">
                <div class="bg-white rounded-lg shadow-lg p-8">
                    <?php if ($show_success || $installation_complete): ?>
                        <div class="text-center">
                            <div class="mx-auto w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mb-6">
                                <svg class="w-8 h-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                </svg>
                            </div>
                            <h1 class="text-3xl font-bold text-gray-900 mb-4">¡Instalación Completada!</h1>
                            <p class="text-gray-600 mb-6">
                                Su sistema de licencias ha sido instalado exitosamente y está listo para usar.
                            </p>
                            
                            <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                                <h4 class="text-red-800 font-semibold mb-2">¡IMPORTANTE - ACCIÓN REQUERIDA!</h4>
                                <p class="text-red-700 text-sm mb-2">
                                    <strong>Elimine inmediatamente el archivo <code>install.php</code> por seguridad.</strong>
                                </p>
                            </div>

                            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6 text-left">
                                <h4 class="text-blue-800 font-semibold mb-2">URLs importantes:</h4>
                                <ul class="text-blue-700 text-sm space-y-1">
                                    <li>• <strong>API:</strong> https://repo.gruposdigitales.com/api/verify.php</li>
                                    <li>• <strong>Panel Admin:</strong> https://repo.gruposdigitales.com/admin/</li>
                                    <li>• <strong>Documentación:</strong> README.md</li>
                                </ul>
                            </div>

                            <a href="admin/" class="inline-block bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors">
                                Acceder al Panel de Administración
                            </a>
                        </div>
                    <?php else: ?>
                        <div class="text-center">
                            <div class="mx-auto w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mb-6">
                                <svg class="w-8 h-8 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                </svg>
                            </div>
                            <h1 class="text-3xl font-bold text-gray-900 mb-4">Error en la Instalación</h1>
                            
                            <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                                <h4 class="text-red-800 font-semibold mb-2">Error:</h4>
                                <p class="text-red-700 text-sm"><?= htmlspecialchars($error) ?></p>
                            </div>

                            <a href="install.php?step=3" class="inline-block bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors">
                                Reintentar
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
}
?> 